<?php
/**
 * Updated Comprehensive Test for Daily Rebate Payout System
 * Auto-detects correct column names and adapts to your database structure
 */

require_once("../../conn.php");
require_once("rebate_functions.php");

// Set timezone
date_default_timezone_set('Asia/Kolkata');

echo "<h2>Daily Rebate Payout System - Updated Comprehensive Test</h2>";

// Test 1: Database connectivity
echo "<h3>Test 1: Database Connectivity</h3>";
if ($conn->connect_error) {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>FAIL:</strong> Database connection failed: " . $conn->connect_error;
    echo "</div>";
    exit;
}

echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
echo "<strong>✓ PASS:</strong> Database connection successful";
echo "</div>";

// Test 2: Table existence
echo "<h3>Test 2: Required Tables</h3>";
$requiredTables = ['bet_rebates', 'vip', 'shonu_kaichila'];
$allTablesExist = true;

foreach ($requiredTables as $table) {
    $tableCheck = $conn->query("SHOW TABLES LIKE '$table'");
    if ($tableCheck->num_rows == 0) {
        echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
        echo "<strong>FAIL:</strong> Table '$table' does not exist";
        echo "</div>";
        $allTablesExist = false;
    } else {
        echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
        echo "<strong>✓ PASS:</strong> Table '$table' exists";
        echo "</div>";
    }
}

if (!$allTablesExist) {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>CRITICAL:</strong> Some required tables are missing. Cannot proceed.";
    echo "</div>";
    exit;
}

// Test 3: Auto-detect column names
echo "<h3>Test 3: Auto-Detect Column Names</h3>";
$tableColumns = [];

foreach ($requiredTables as $table) {
    $structureQuery = "DESCRIBE $table";
    $structureResult = $conn->query($structureQuery);
    
    if ($structureResult) {
        $columns = [];
        while ($row = $structureResult->fetch_assoc()) {
            $columns[] = $row['Field'];
        }
        $tableColumns[$table] = $columns;
        
        echo "<div style='background: #e7f3ff; padding: 10px; border-radius: 5px; color: #004085; margin: 5px 0;'>";
        echo "<strong>📋 Table '$table' columns:</strong> " . implode(', ', $columns);
        echo "</div>";
    }
}

// Auto-detect correct column names
$detectedColumns = [
    'user_id_vip' => null,
    'vip_level_vip' => null,
    'user_id_wallet' => null,
    'wallet_balance' => null
];

// Check VIP table for user ID column
if (isset($tableColumns['vip'])) {
    $vipColumns = $tableColumns['vip'];
    if (in_array('user_id', $vipColumns)) {
        $detectedColumns['user_id_vip'] = 'user_id';
    } elseif (in_array('userid', $vipColumns)) {
        $detectedColumns['user_id_vip'] = 'userid';
    } elseif (in_array('id', $vipColumns)) {
        $detectedColumns['user_id_vip'] = 'id';
    }
    
    // Check VIP level column
    if (in_array('vip_level', $vipColumns)) {
        $detectedColumns['vip_level_vip'] = 'vip_level';
    } elseif (in_array('lvl', $vipColumns)) {
        $detectedColumns['vip_level_vip'] = 'lvl';
    } elseif (in_array('level', $vipColumns)) {
        $detectedColumns['vip_level_vip'] = 'level';
    }
}

// Check shonu_kaichila table for user ID and wallet balance
if (isset($tableColumns['shonu_kaichila'])) {
    $walletColumns = $tableColumns['shonu_kaichila'];
    
    // Check for user ID column - based on your structure, it's 'balakedara'
    if (in_array('user_id', $walletColumns)) {
        $detectedColumns['user_id_wallet'] = 'user_id';
    } elseif (in_array('userid', $walletColumns)) {
        $detectedColumns['user_id_wallet'] = 'userid';
    } elseif (in_array('balakedara', $walletColumns)) {
        $detectedColumns['user_id_wallet'] = 'balakedara';  // Your actual user_id column
    } elseif (in_array('id', $walletColumns)) {
        $detectedColumns['user_id_wallet'] = 'id';
    }
    
    // Check wallet balance column - based on your structure, it's likely 'motta'
    if (in_array('wallet_balance', $walletColumns)) {
        $detectedColumns['wallet_balance'] = 'wallet_balance';
    } elseif (in_array('balance', $walletColumns)) {
        $detectedColumns['wallet_balance'] = 'balance';
    } elseif (in_array('wallet', $walletColumns)) {
        $detectedColumns['wallet_balance'] = 'wallet';
    } elseif (in_array('motta', $walletColumns)) {
        $detectedColumns['wallet_balance'] = 'motta';  // Your actual wallet balance column
    } elseif (in_array('amount', $walletColumns)) {
        $detectedColumns['wallet_balance'] = 'amount';
    } elseif (in_array('rebet', $walletColumns)) {
        $detectedColumns['wallet_balance'] = 'rebet';  // Alternative balance column
    }
}

// Display detected columns
echo "<h4>Auto-Detected Column Mappings:</h4>";
foreach ($detectedColumns as $purpose => $columnName) {
    if ($columnName) {
        echo "<div style='background: #d4edda; padding: 5px; border-radius: 3px; color: #155724; margin: 2px;'>";
        echo "<strong>✓ DETECTED:</strong> $purpose → $columnName";
        echo "</div>";
    } else {
        echo "<div style='background: #f8d7da; padding: 5px; border-radius: 3px; color: #721c24; margin: 2px;'>";
        echo "<strong>✗ NOT FOUND:</strong> $purpose";
        echo "</div>";
    }
}

// Test 4: VIP Rate Functions
echo "<h3>Test 4: VIP Rate Functions</h3>";
$vipRates = [
    0 => 0.0000,  // No VIP
    1 => 0.0005,  // VIP 1
    2 => 0.0008,  // VIP 2
    3 => 0.0012,  // VIP 3
    4 => 0.0015,  // VIP 4
    5 => 0.0018,  // VIP 5
    6 => 0.0020,  // VIP 6
    7 => 0.0022,  // VIP 7
    8 => 0.0025,  // VIP 8
    9 => 0.0028,  // VIP 9
    10 => 0.0030  // VIP 10
];

$allRatesCorrect = true;
foreach ($vipRates as $vipLevel => $expectedRate) {
    $actualRate = getVipRebateRate($vipLevel);
    if (abs($actualRate - $expectedRate) < 0.0001) {
        echo "<div style='background: #d4edda; padding: 3px; border-radius: 3px; color: #155724; margin: 1px;'>";
        echo "<strong>✓ PASS:</strong> VIP $vipLevel rate: " . ($actualRate * 100) . "%";
        echo "</div>";
    } else {
        echo "<div style='background: #f8d7da; padding: 3px; border-radius: 3px; color: #721c24; margin: 1px;'>";
        echo "<strong>FAIL:</strong> VIP $vipLevel rate: Expected " . ($expectedRate * 100) . "%, Got " . ($actualRate * 100) . "%";
        echo "</div>";
        $allRatesCorrect = false;
    }
}

if ($allRatesCorrect) {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✓ PASS:</strong> All VIP rate functions work correctly";
    echo "</div>";
}

// Test 5: Sample Data Retrieval with Auto-Detected Columns
echo "<h3>Test 5: Sample Data Retrieval with Auto-Detected Columns</h3>";

// Check bet_rebates table columns
$betRebatesColumns = [];
if (isset($tableColumns['bet_rebates'])) {
    $betRebatesColumns = $tableColumns['bet_rebates'];
}

$hasPaidColumn = in_array('is_paid', $betRebatesColumns);
$hasPaidAtColumn = in_array('paid_at', $betRebatesColumns);

if ($detectedColumns['user_id_wallet'] && $detectedColumns['wallet_balance']) {
    try {
        // Build query with detected column names
        $userIdWallet = $detectedColumns['user_id_wallet'];
        $walletBalance = $detectedColumns['wallet_balance'];
        
        $testQuery = "SELECT 
                        br.id,
                        br.user_id,
                        br.rebate_amount,
                        sk.$walletBalance as wallet_balance,
                        DATE(br.created_at) as rebate_date
                      FROM bet_rebates br
                      JOIN shonu_kaichila sk ON br.user_id = sk.$userIdWallet";
        
        if ($hasPaidColumn) {
            $testQuery .= " WHERE (br.is_paid IS NULL OR br.is_paid = 0)";
        }
        
        $testQuery .= " ORDER BY br.created_at DESC LIMIT 5";
        
        $testResult = $conn->query($testQuery);
        
        if ($testResult) {
            echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
            echo "<strong>✓ PASS:</strong> Query executed successfully";
            echo "</div>";
            
            if ($testResult->num_rows > 0) {
                echo "<h4>Sample Rebate Records:</h4>";
                echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
                echo "<tr style='background: #f2f2f2;'>";
                echo "<th>ID</th><th>User ID</th><th>Rebate Amount</th><th>Wallet Balance</th><th>Date</th>";
                echo "</tr>";
                
                while ($row = $testResult->fetch_assoc()) {
                    echo "<tr>";
                    echo "<td>" . htmlspecialchars($row['id']) . "</td>";
                    echo "<td>" . htmlspecialchars($row['user_id']) . "</td>";
                    echo "<td>₹" . number_format($row['rebate_amount'], 2) . "</td>";
                    echo "<td>₹" . number_format($row['wallet_balance'], 2) . "</td>";
                    echo "<td>" . htmlspecialchars($row['rebate_date']) . "</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<div style='background: #fff3cd; padding: 10px; border-radius: 5px; color: #856404;'>";
                echo "<strong>INFO:</strong> No rebate records found";
                echo "</div>";
            }
        } else {
            echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
            echo "<strong>FAIL:</strong> Query failed: " . $conn->error;
            echo "</div>";
        }
    } catch (Exception $e) {
        echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
        echo "<strong>ERROR:</strong> " . $e->getMessage();
        echo "</div>";
    }
} else {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>FAIL:</strong> Cannot build query - missing required column mappings";
    echo "</div>";
}

// Test 6: VIP User Test
echo "<h3>Test 6: VIP User Test</h3>";
if ($detectedColumns['user_id_vip'] && $detectedColumns['vip_level_vip']) {
    $userIdVip = $detectedColumns['user_id_vip'];
    $vipLevelVip = $detectedColumns['vip_level_vip'];
    
    $vipTestQuery = "SELECT $userIdVip as user_id, $vipLevelVip as vip_level FROM vip ORDER BY $vipLevelVip DESC LIMIT 5";
    $vipTestResult = $conn->query($vipTestQuery);
    
    if ($vipTestResult && $vipTestResult->num_rows > 0) {
        echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
        echo "<strong>✓ PASS:</strong> VIP users found";
        echo "</div>";
        
        echo "<h4>Sample VIP Users:</h4>";
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr style='background: #f2f2f2;'>";
        echo "<th>User ID</th><th>VIP Level</th><th>Rebate Rate</th>";
        echo "</tr>";
        
        while ($row = $vipTestResult->fetch_assoc()) {
            $vipLevel = (int)$row['vip_level'];
            $rebateRate = getVipRebateRate($vipLevel);
            
            echo "<tr>";
            echo "<td>" . htmlspecialchars($row['user_id']) . "</td>";
            echo "<td>" . htmlspecialchars($vipLevel) . "</td>";
            echo "<td>" . ($rebateRate * 100) . "%</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<div style='background: #fff3cd; padding: 10px; border-radius: 5px; color: #856404;'>";
        echo "<strong>INFO:</strong> No VIP users found";
        echo "</div>";
    }
} else {
    echo "<div style='background: #f8d7da; padding: 10px; border-radius: 5px; color: #721c24;'>";
    echo "<strong>FAIL:</strong> Cannot test VIP users - missing VIP column mappings";
    echo "</div>";
}

// Test 7: Generate Corrected Migration Script
echo "<h3>Test 7: Generate Corrected Migration Script</h3>";
$migrationNeeded = false;
$migrationScript = "<?php\n";
$migrationScript .= "// Auto-generated migration script based on detected database structure\n";
$migrationScript .= "require_once('../../conn.php');\n\n";

// Check if bet_rebates needs is_paid and paid_at columns
if (!$hasPaidColumn || !$hasPaidAtColumn) {
    $migrationNeeded = true;
    $migrationScript .= "// Add payment tracking columns to bet_rebates\n";
    if (!$hasPaidColumn) {
        $migrationScript .= "\$conn->query(\"ALTER TABLE bet_rebates ADD COLUMN is_paid TINYINT(1) DEFAULT 0\");\n";
    }
    if (!$hasPaidAtColumn) {
        $migrationScript .= "\$conn->query(\"ALTER TABLE bet_rebates ADD COLUMN paid_at TIMESTAMP NULL\");\n";
    }
    $migrationScript .= "\n";
}

// Check if bet_rebates needs vip_level column
if (!in_array('vip_level', $betRebatesColumns)) {
    $migrationNeeded = true;
    $migrationScript .= "// Add VIP level column to bet_rebates\n";
    $migrationScript .= "\$conn->query(\"ALTER TABLE bet_rebates ADD COLUMN vip_level INT DEFAULT 0\");\n\n";
}

$migrationScript .= "echo 'Migration completed successfully!';\n";
$migrationScript .= "?>";

if ($migrationNeeded) {
    echo "<div style='background: #fff3cd; padding: 10px; border-radius: 5px; color: #856404;'>";
    echo "<strong>⚠️ MIGRATION NEEDED:</strong> Some columns are missing";
    echo "</div>";
    
    echo "<h4>Generated Migration Script:</h4>";
    echo "<textarea style='width: 100%; height: 200px; font-family: monospace;'>";
    echo htmlspecialchars($migrationScript);
    echo "</textarea>";
    
    // Save the migration script
    file_put_contents('auto_migrate_rebate_table.php', $migrationScript);
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✓ SAVED:</strong> Migration script saved as auto_migrate_rebate_table.php";
    echo "</div>";
} else {
    echo "<div style='background: #d4edda; padding: 10px; border-radius: 5px; color: #155724;'>";
    echo "<strong>✓ PASS:</strong> No migration needed";
    echo "</div>";
}

// Test 8: System Status Summary
echo "<h3>Test 8: System Status Summary</h3>";
$systemStatus = [
    'database' => !$conn->connect_error,
    'tables' => $allTablesExist,
    'columns_detected' => ($detectedColumns['user_id_wallet'] && $detectedColumns['wallet_balance']),
    'vip_rates' => $allRatesCorrect,
    'permissions' => is_writable('.')
];

$allPassed = true;
foreach ($systemStatus as $test => $passed) {
    if (!$passed) {
        $allPassed = false;
        break;
    }
}

if ($allPassed) {
    echo "<div style='background: #d4edda; padding: 15px; border-radius: 5px; color: #155724; border: 2px solid #28a745;'>";
    echo "<strong>🎉 SYSTEM READY!</strong><br>";
    echo "Your Daily Rebate Payout System is configured for your database!<br>";
    echo "✓ Database connectivity: OK<br>";
    echo "✓ Required tables: OK<br>";
    echo "✓ Column mappings: Auto-detected<br>";
    echo "✓ VIP rate functions: OK<br>";
    echo "✓ File permissions: OK<br>";
    echo "<br><strong>Column Mappings Detected:</strong><br>";
    foreach ($detectedColumns as $purpose => $columnName) {
        if ($columnName) {
            echo "• $purpose → $columnName<br>";
        }
    }
    echo "</div>";
} else {
    echo "<div style='background: #fff3cd; padding: 15px; border-radius: 5px; color: #856404; border: 2px solid #ffc107;'>";
    echo "<strong>⚠️ SYSTEM NEEDS ATTENTION!</strong><br>";
    echo "Some issues were found but the system can work with adaptations.<br>";
    if ($migrationNeeded) {
        echo "• Run auto_migrate_rebate_table.php to add missing columns<br>";
    }
    echo "• Check the detected column mappings above<br>";
    echo "• Test with a small batch of users first";
    echo "</div>";
}

$conn->close();
?>

<style>
body {
    font-family: Arial, sans-serif;
    max-width: 1200px;
    margin: 0 auto;
    padding: 20px;
    background-color: #f8f9fa;
}

h2 {
    color: #333;
    border-bottom: 2px solid #007bff;
    padding-bottom: 10px;
}

h3 {
    color: #555;
    margin-top: 30px;
    margin-bottom: 15px;
}

table {
    margin: 10px 0;
    background: white;
    border-radius: 5px;
    overflow: hidden;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

th {
    background-color: #007bff;
    color: white;
    padding: 12px;
    text-align: left;
}

td {
    padding: 8px 12px;
    border-bottom: 1px solid #eee;
}

tr:hover {
    background-color: #f8f9fa;
}

textarea {
    border: 1px solid #ddd;
    border-radius: 4px;
    padding: 10px;
}
</style>
